package ihm;

import java.io.IOException;
import java.net.URL;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;
import javafx.application.Platform;
import javafx.embed.swing.JFXPanel;
import javafx.event.ActionEvent;
import javafx.fxml.FXML;
import javafx.fxml.FXMLLoader;
import javafx.fxml.Initializable;
import javafx.scene.Parent;
import javafx.scene.Scene;
import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.stage.Modality;
import javafx.stage.Stage;
import javafx.util.Callback;
import javax.swing.SwingUtilities;
import beans.Person;
import ctrl.ICtrlIhm;

/**
 * Modèle d'application "MVC2" réalisé avec interfaces (vues au 226a).
 * <p>
 * Cette classe représente l'Ihm principale l'application MVC2.
 * <p>
 * Cette Ihm est en JavaFx. Le code qu'on trouve ci-dessous est donc celle du
 * "contrôleur de notre vue", vue dont le contenu est lui exprimé séparément en
 * FXML.
 *
 * @author EMF-Informatique
 */
public class Ihm implements Initializable, IIhmCtrl {

    /**
     * Constructeur de notre IHM ou l'on initalize le chemin vers la vue.
     */
    public Ihm() {
        //Si vous avez une erreur "Location is not set" c'est que ce chemin est faux.
        fxml = "/ihm/Ihm.fxml";
        refCtrl = null;
    }

    /**
     * Permet de démarrer la GUI JavaFX qui se trouve dans le chemin stocké dans
     * fxml
     * Tout ceci pour atteindre le dispatcher thread qui est obligatoire pour
     * JavaFX.
     */
    @Override
    @SuppressWarnings( "ResultOfObjectAllocationIgnored" )
    public void start() {
        Ihm myself = this;

        //Callback pour le ControllerFactory, quand JavaFX voudra créer le controlleur de vue, viendra dans ce callback et donnera l'instance déjà créée au lieu d'en faire une nouvelle.
        Callback<Class<?>, Object> controllerFactory = type -> {
            return myself;
        };

        //Commence l'initialisation dans le thread de java 
        SwingUtilities.invokeLater( () -> {
            new JFXPanel(); // Permet d'initializer le toolkit et l'environment JavaFX
            //Accède au dispatcher thread tant aimé par JavaFX
            Platform.runLater( () -> {
                try {
                    Stage stage = new Stage();
                    FXMLLoader fxmlLoader = new FXMLLoader( getClass().getResource( fxml ) );

                    fxmlLoader.setControllerFactory( controllerFactory );

                    Parent root = ( Parent ) fxmlLoader.load();

                    Scene scene = new Scene( root );

                    stage.setScene( scene );
                    stage.setTitle( "Simple MVC2" );
                    stage.show();
                }
                catch ( IOException ex ) {
                    System.out.println( "Can't start the IHM because : " + ex );
                    Platform.exit();
                }
            } );
        } );
    }

    /**
     * La méthode appellée quand la vue est initialisée
     */
    @Override
    public void initialize( URL url, ResourceBundle rb ) {
        indexOfPerson = 0;
        currentPerson = refCtrl.getPerson( indexOfPerson );
        displayCurrentPerson();
        btnLeft.setDisable( true );
    }

    /**
     * Méthode appelée lors de la pression sur le bouton édit.
     *
     * @param event l'évènement du bouton édit.
     */
    @FXML
    private void editPerson( ActionEvent event ) {
        Stage childStage = openChildWindow();
        EditIhm editIhm = ( EditIhm ) childStage.getUserData();
        editIhm.setPerson( currentPerson );
        currentPerson = refCtrl.getPerson( indexOfPerson );
        displayCurrentPerson();
    }

    /**
     * Méthode appelée lors de la pression sur le bouton précédent.
     *
     * @param event l'évènement du bouton précédent.
     */
    @FXML
    private void previous( ActionEvent event ) {
        indexOfPerson--;
        currentPerson = refCtrl.getPerson( indexOfPerson );
        displayCurrentPerson();
        changeButtonState();
    }

    /**
     * Méthode appelée lors de la pression sur le bouton suivant.
     *
     * @param event l'évènement du bouton suivant.
     */
    @FXML
    private void next( ActionEvent event ) {
        indexOfPerson++;
        currentPerson = refCtrl.getPerson( indexOfPerson );
        displayCurrentPerson();
        changeButtonState();
    }

    /**
     * Méthode pour afficher la personne courante.
     */
    private void displayCurrentPerson() {
        lblLastname.textProperty().bind( currentPerson.lastnameProperty() );
        lblFirstname.textProperty().bind( currentPerson.firstnameProperty() );
        lblBirthDate.textProperty().bind( currentPerson.birthDateProperty().asString() );
        lblAge.textProperty().bind( currentPerson.ageProperty().asString() );
    }

    /**
     * Méthode pour activer ou desactiver les boutons suivant/précédents.
     */
    private void changeButtonState() {
        if ( refCtrl.isNextAvailable( indexOfPerson ) ) {
            btnRight.setDisable( false );
        } else {
            btnRight.setDisable( true );
        }

        if ( refCtrl.isPreviousAvailable( indexOfPerson ) ) {
            btnLeft.setDisable( false );
        } else {
            btnLeft.setDisable( true );
        }
    }

    /**
     * Méthode qui permet de lancer une fenêtre enfant.
     *
     * @return Le stage de la fenêtre.
     */
    private Stage openChildWindow() {
        Stage childStage = null;
        try {
            FXMLLoader fxmlLoader = new FXMLLoader( getClass().getResource( "/ihm/EditIhm.fxml" ) );
            Parent root = ( Parent ) fxmlLoader.load();
            EditIhm editIhm = fxmlLoader.getController();
            childStage = new Stage();
            childStage.initModality( Modality.WINDOW_MODAL );
            childStage.initOwner( btnEdit.getScene().getWindow() );
            childStage.setTitle( "Edit Value" );
            childStage.setUserData( editIhm );
            Scene scene = new Scene( root );
            childStage.setScene( scene );
            childStage.show();
        }
        catch ( IOException ex ) {
            Logger.getLogger( Ihm.class.getName() ).log( Level.SEVERE, null, ex );
        }
        return childStage;
    }

    public ICtrlIhm getRefCtrl() {
        return refCtrl;
    }

    public void setRefCtrl( ICtrlIhm refCtrl ) {
        this.refCtrl = refCtrl;
    }

    @FXML
    private Label lblLastname;
    @FXML
    private Label lblFirstname;
    @FXML
    private Label lblBirthDate;
    @FXML
    private Label lblAge;
    @FXML
    private Button btnLeft;
    @FXML
    private Button btnEdit;
    @FXML
    private Button btnRight;

    private ICtrlIhm refCtrl;
    private final String fxml;
    private int indexOfPerson;
    private Person currentPerson;

}
